/******************************************************************************

  Copyright (c) 2001-2017, Intel Corporation
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.

   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   3. Neither the name of the Intel Corporation nor the names of its
      contributors may be used to endorse or promote products derived from
      this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.

******************************************************************************/
/*$FreeBSD$*/

#include "ixgbe.h"

inline u16
ixgbe_read_pci_cfg_pf(struct ixgbe_hw *hw, u32 reg)
{
	return pci_read_config(((struct ixgbe_softc *)hw->back)->dev, reg, 2);
}

inline void
ixgbe_write_pci_cfg_pf(struct ixgbe_hw *hw, u32 reg, u16 value)
{
	pci_write_config(((struct ixgbe_softc *)hw->back)->dev, reg, value, 2);
}

inline u32
ixgbe_read_reg_pf(struct ixgbe_hw *hw, u32 reg)
{
	struct ixgbe_softc *sc = (struct ixgbe_softc *)hw->back;
	u32 retval;
	u8 i;

	retval = bus_space_read_4(sc->osdep.mem_bus_space_tag,
	    sc->osdep.mem_bus_space_handle, reg);

	/* Normal... */
	if ((retval != 0xDEADBEEF) ||
	    !(hw->phy.nw_mng_if_sel & IXGBE_NW_MNG_IF_SEL_SGMII_ENABLE))
		return retval;

	/* Unusual... */

	/*
	 * 10/100 Mb mode has a quirk where it's possible the previous
	 * write to the Phy hasn't completed.  So we keep trying.
	 */
	for (i = 100; retval; i--) {
		if (!i) {
			device_printf(sc->dev, "Register (0x%08X) writes did not complete: 0x%08X\n",
			    reg, retval);
			break;
		}
		retval = bus_space_read_4(sc->osdep.mem_bus_space_tag,
		    sc->osdep.mem_bus_space_handle, IXGBE_MAC_SGMII_BUSY);
	}

	for (i = 10; retval == 0xDEADBEEF; i--) {
		if (!i) {
			device_printf(sc->dev,
			    "Failed to read register 0x%08X.\n", reg);
			break;
		}
		retval = bus_space_read_4(sc->osdep.mem_bus_space_tag,
		    sc->osdep.mem_bus_space_handle, reg);
	}

	return retval;
}

inline void
ixgbe_write_reg_pf(struct ixgbe_hw *hw, u32 reg, u32 val)
{
	bus_space_write_4(((struct ixgbe_softc *)hw->back)->osdep.mem_bus_space_tag,
	    ((struct ixgbe_softc *)hw->back)->osdep.mem_bus_space_handle,
	    reg, val);
}

inline u32
ixgbe_read_reg_array_pf(struct ixgbe_hw *hw, u32 reg, u32 offset)
{
	return bus_space_read_4(((struct ixgbe_softc *)hw->back)->osdep.mem_bus_space_tag,
	    ((struct ixgbe_softc *)hw->back)->osdep.mem_bus_space_handle,
	    reg + (offset << 2));
}

inline void
ixgbe_write_reg_array_pf(struct ixgbe_hw *hw, u32 reg, u32 offset, u32 val)
{
	bus_space_write_4(((struct ixgbe_softc *)hw->back)->osdep.mem_bus_space_tag,
	    ((struct ixgbe_softc *)hw->back)->osdep.mem_bus_space_handle,
	    reg + (offset << 2), val);
}

void
ixgbe_init_lock(struct ixgbe_lock *lock)
{
	mtx_init(&lock->mutex, "mutex",
	    "ixgbe ACI lock", MTX_DEF | MTX_DUPOK);
}

void
ixgbe_acquire_lock(struct ixgbe_lock *lock)
{
	mtx_lock(&lock->mutex);
}

void
ixgbe_release_lock(struct ixgbe_lock *lock)
{
	mtx_unlock(&lock->mutex);
}

void
ixgbe_destroy_lock(struct ixgbe_lock *lock)
{
	if (mtx_initialized(&lock->mutex))
		mtx_destroy(&lock->mutex);
}

/**
 * ixgbe_info_fwlog - Format and print an array of values to the console
 * @hw: private hardware structure
 * @rowsize: preferred number of rows to use
 * @groupsize: preferred size in bytes to print each chunk
 * @buf: the array buffer to print
 * @len: size of the array buffer
 *
 * Format the given array as a series of uint8_t values with hexadecimal
 * notation and log the contents to the console log.  This variation is
 * specific to firmware logging.
 *
 * TODO: Currently only supports a group size of 1, due to the way hexdump is
 * implemented.
 */
void
ixgbe_info_fwlog(struct ixgbe_hw *hw, uint32_t rowsize, uint32_t __unused groupsize,
	       uint8_t *buf, size_t len)
{
	device_t dev = ((struct ixgbe_softc *)hw->back)->dev;
	char prettyname[20];

	if (!ixgbe_fwlog_supported(hw))
		return;

	/* Format the device header to a string */
	snprintf(prettyname, sizeof(prettyname), "%s: FWLOG: ",
	    device_get_nameunit(dev));

	/* Make sure the row-size isn't too large */
	if (rowsize > 0xFF)
		rowsize = 0xFF;

	hexdump(buf, len, prettyname, HD_OMIT_CHARS | rowsize);
}

