/* SPDX-License-Identifier: BSD-3-Clause */
/*  Copyright (c) 2025, Intel Corporation
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *
 *   3. Neither the name of the Intel Corporation nor the names of its
 *      contributors may be used to endorse or promote products derived from
 *      this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 *  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 *  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 *  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 *  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *  POSSIBILITY OF SUCH DAMAGE.
 */
/*$FreeBSD$*/

/**
 * @file freebsd_compat_common.h
 * @brief FreeBSD kernel compatibility macros used by Intel drivers
 *
 * Contains macros and function backports used to enable compatibility with
 * a variety of FreeBSD kernel versions.
 *
 * iflib or legacy driver specific compatibility definitions can be found in
 * the freebsd_compat_iflib.[ch] and freebsd_compat_legacy.[ch] files.
 */
#ifndef _FREEBSD_COMPAT_COMMON_H_
#define _FREEBSD_COMPAT_COMMON_H_

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <net/if.h>
#include <net/if_var.h>

#ifndef ETHER_IS_ZERO
#define	ETHER_IS_ZERO(addr) \
	(((addr)[0] | (addr)[1] | (addr)[2] | \
	  (addr)[3] | (addr)[4] | (addr)[5]) == 0x00)
#endif

#ifndef USEC_2_TICKS
#define USEC_2_TICKS(u) max(1, (uint32_t)((hz == 1000) ? \
	  ((u) / 1000) : ((uint64_t)(u) * (uint64_t)hz)/(uint64_t)1000000))
#endif
#ifndef MSEC_2_TICKS
#define MSEC_2_TICKS(m) max(1, (uint32_t)((hz == 1000) ? \
	  (m) : ((uint64_t)(m) * (uint64_t)hz)/(uint64_t)1000))
#endif
#ifndef TICKS_2_MSEC
#define TICKS_2_MSEC(t) max(1, (uint32_t)(hz == 1000) ? \
          (t) : (((uint64_t)(t) * (uint64_t)1000)/(uint64_t)hz))
#endif
#ifndef TICKS_2_USEC
#define TICKS_2_USEC(t) max(1, (uint32_t)(hz == 1000) ? \
          ((t) * 1000) : (((uint64_t)(t) * (uint64_t)1000000)/(uint64_t)hz))
#endif

/*
 * Old versions of <sys/bitstring.h> lack bit_ffs_area_at and bit_ffc_area_at.
 * Indicate that drivers should use the bitstring_compat.h when building on
 * these older kernels.
 */
#if __FreeBSD_version < 1300061
#define USE_BITSTRING_COMPAT
#endif

/*
 * Newer versions of FreeBSD introduced new KPI for accessing lists of
 * multicast addresses. The old KPI is eventually removed in 1300054, so we
 * provide the new implementation for older kernels that lack them.
 */
#if __FreeBSD_version < 1300051
/* Part of the new KPI was MFCed to FreeBSD 12.2 */
#if (__FreeBSD_version < 1202000 || __FreeBSD_version > 1300000 )
struct sockaddr_dl;
typedef u_int iflladdr_cb_t(void *, struct sockaddr_dl *, u_int);
u_int if_foreach_lladdr(if_t ifp, iflladdr_cb_t cb, void *cb_arg);
u_int if_foreach_llmaddr(if_t ifp, iflladdr_cb_t cb, void *cb_arg);
#endif
u_int if_lladdr_count(if_t ifp);
u_int if_llmaddr_count(if_t ifp);
#endif

/*
 * FreeBSD 14.0 changed the macro arguments to DRIVER_MODULE in order to remove
 * the unused devclass argument.
 */
#if __FreeBSD_version >= 1400066
#define COMPAT_DRIVER_MODULE(_name, _busname, _driver, _evh) \
	DRIVER_MODULE(_name, _busname, _driver, _evh, 0)
#else
#define COMPAT_DRIVER_MODULE(_name, _busname, _driver, _evh) \
	devclass_t _name##_devclass; \
	DRIVER_MODULE(_name, _busname, _driver, _name##_devclass, _evh, 0)
#endif

/*
 * FreeBSD 13.0 introduced EPOCH aware task queues.
 * This dummy wrapper is defined for 12.X branch
 * to avoid conditional compilation in base drivers' code.
 */
#ifndef NET_TASK_INIT
#define NET_TASK_INIT(t, p, f, c) TASK_INIT(t, p, f, c)
#endif

/*
 * IFF_KNOWSEPOCH flag is used to let the stack know that driver is EPOCH aware.
 * Define IFF_KNOWSEPOCH for versions before FreeBSD 13.0.
 */
#ifndef IFF_KNOWSEPOCH
#define IFF_KNOWSEPOCH 0
#endif

#endif /* _FREEBSD_COMPAT_COMMON_H_ */
