/* SPDX-License-Identifier: BSD-3-Clause */
/*  Copyright (c) 2025, Intel Corporation
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *
 *   3. Neither the name of the Intel Corporation nor the names of its
 *      contributors may be used to endorse or promote products derived from
 *      this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 *  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 *  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 *  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 *  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *  POSSIBILITY OF SUCH DAMAGE.
 */
/*$FreeBSD$*/

/**
 * @file freebsd_compat_iflib.h
 * @brief FreeBSD kernel compatibility macros for iflib network drivers
 *
 * Contains macro definitions and function backports to aid in developing
 * an iflib network driver that compiles on a variety of FreeBSD kernels.
 *
 * For generic compatibility macros used by both iflib and legacy network
 * drivers, see freebsd_compat_common.[ch]
 */
#ifndef _FREEBSD_COMPAT_IFLIB_H_
#define _FREEBSD_COMPAT_IFLIB_H_

#include "freebsd_compat_common.h"

#include <net/ethernet.h>
#include <net/iflib.h>
#include "ifdi_if.h"

/*
 * Over the course of its development, iflib has made a few API changes, which
 * we need to work around here in order to allow the driver code to work on
 * different versions of the FreeBSD kernel.
 */

#if __FreeBSD_version >= 1200073
#define HAVE_TSO_MAX_IN_SHARED_CTX
#define HAVE_CAPABILITIES_IN_SOFTC_CTX
#endif

#if (__FreeBSD_version >= 1200085 || \
     (__FreeBSD_version < 1200000 && __FreeBSD_version >= 1103000))
#define HAVE_IFLIB_IN_DETACH
#endif

#if __FreeBSD_version < 1200000
#define IFLIB_CTX_LOCK_IS_MTX
#else
#define IFLIB_ATTACH_HOLDS_CTX_LOCK
#endif

#if __FreeBSD_version < 1200516
/**
 * iflib_device_probe_vendor - Probe a device, returning BUS_PROBE_VENDOR
 * @dev: the device to probe
 *
 * Calls iflib_device_probe to determine if the given device is valid for this
 * driver. On success, instead of BUS_PROBE_DEFAULT, it will report
 * BUS_PROBE_VENDOR. This is used to indicate that the driver is a vendor
 * driver and should have higher priority than the in-kernel driver.
 *
 * @returns BUS_PROBE_VENDOR or an error indicating the device cannot load
 * with this driver.
 */
static inline int
iflib_device_probe_vendor(device_t dev)
{
	int probe;

	probe = iflib_device_probe(dev);
	if (probe == BUS_PROBE_DEFAULT)
		return (BUS_PROBE_VENDOR);
	else
		return (probe);
}
#endif
#endif /* _FREEBSD_COMPAT_IFLIB_H_ */
